/* ------------------------------------------------------------------------------
  File: chr6dm_packet_handler.c
  Author: CH Robotics
  Version: 1.0
  
  Description: Functions used to handle packets sent and received by the CHR-6dm
------------------------------------------------------------------------------ */ 

#include "stm32f10x.h"
#include "CHR_packet_handler.h"
#include "UM6_config.h"
#include "UM6_states.h"

/*******************************************************************************
* Function Name  : ProcessPacket
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Handles packets received over the UART.  First, the packet is checked to make
sure it is understood.  If not, a packet is transmitted over the USART to that
effect.  If it is, then the following takes place:

1. If the packet is a write operation, then the data contained in data section
of the packet is copied to the relevant registers.
2. If the packet is a read operation, the relevant registers are copied into
a new packet.  The new packet is then transmitted.
3. Finally, the packet is sent to the DispatchPacket(.) function.  The DispatchPacket
function is used to handle packet-specific actions that must be performed when the
packet is received.  For example, a packet that alters the broadcast rate of the
device must change the broadcast timer configuration to reflect the change.  This
is handled by the DispatchPacket function.
						 
*******************************************************************************/
void ProcessPacket( USARTPacket* new_packet )
{
	 uint8_t address_valid = 1;
	 uint8_t batch_good = 1;
	 	 
	 // Check to make sure that the packet address is recognizable.  For example, if it involves
	 // a read or a write to the configuration array, then the address must be less than CONFIG_ARRAY_SIZE,
	 // which is defined in UM6_config.h.  Similarly, a read or write to the data array must be to an
	 // address lower than DATA_ARRAY_SIZE, and a command packet must have an address lower than COMMAND_COUNT
	 // (also defined in UM6_config.h)
	 
	 // Start with a check for configuration register reads/writes.  Note that valid configuration registers have addresses
	 // starting at CONFIG_REG_START_ADDRESS and ending at DATA_REG_START_ADDRESS - 1 (this is based on how the communication
	 // protocol is defined)
	 if( new_packet->address_type == ADDRESS_TYPE_CONFIG )
	 {
		  if( (new_packet->address-CONFIG_REG_START_ADDRESS) >= CONFIG_ARRAY_SIZE )
		  {
				address_valid = 0;
		  }
		  
		  // Check if this is a batch operation and, if so, whether it is valid (cannot allow a batch operation to go beyond array bounds)
		  if( new_packet->PT & PACKET_IS_BATCH )
		  {
				if( (new_packet->address - CONFIG_REG_START_ADDRESS + ((new_packet->PT >> 2) & PACKET_BATCH_LENGTH_MASK) - 1) >= CONFIG_ARRAY_SIZE )
				{
					 batch_good = 0;
				}
		  }
	 }
	 // Check for invalid data register addresses now...
	 else if( new_packet->address_type == ADDRESS_TYPE_DATA )
	 {
		  if( (new_packet->address-DATA_REG_START_ADDRESS) >= DATA_ARRAY_SIZE )
		  {
				address_valid = 0;
		  }
		  
		  // Check if this is a batch operation and, if so, whether it is valid (cannot allow a batch operation to go beyond array bounds)
		  if( new_packet->PT & PACKET_IS_BATCH )
		  {
				if( (new_packet->address - DATA_REG_START_ADDRESS + ((new_packet->PT >> 2) & PACKET_BATCH_LENGTH_MASK) - 1) >= DATA_ARRAY_SIZE )
				{
					 batch_good = 0;
				}
		  }
	 }
	 // Now check for invalid commands
	 else if( new_packet->address_type == ADDRESS_TYPE_COMMAND )
	 {
		  if( (new_packet->address - COMMAND_START_ADDRESS) >= COMMAND_COUNT )
		  {
				address_valid = 0;
		  }
	 }
	 
	 // If the address check failed, send a packet informing about the problem
	 if( !address_valid )
	 {
		  USARTPacket response_packet;
		  
		  // Send "unknown address" packet
		  response_packet.PT = PACKET_NO_DATA;
		  response_packet.address = UM6_UNKNOWN_ADDRESS;
		  response_packet.data_length = 0;	// No data bytes
		  response_packet.checksum = ComputeChecksum( &response_packet );
		  
		  SendTXPacketSafe( &response_packet );
		  
		  return;
	 }

	 // If the the batch size was too large to be valid, send a packet informing about the problem
	 if( !batch_good )
	 {
		  USARTPacket response_packet;
		  
		  // Send "invalid batch size" packet
		  response_packet.PT = PACKET_NO_DATA;
		  response_packet.address = UM6_INVALID_BATCH_SIZE;
		  response_packet.data_length = 0;	// No data bytes
		  response_packet.checksum = ComputeChecksum( &response_packet );
		  
		  SendTXPacketSafe( &response_packet );
		  
		  return;
	 }
	 
	 // Now process the packet.  
	 // Only read and write packets will be processed in this function.  Commands will be handled later, in the call to DispatchPacket(.)
	 if( new_packet->address_type != ADDRESS_TYPE_COMMAND )
	 {
		  // If the packet is performing a write operation, copy the data to the specified address(es)
		  // If it is a batch write operation, then multiple registers will be written.
		  if( new_packet->PT & PACKET_HAS_DATA )
		  {
				int i;
				int address_offset = 0;
				
				// Received packets will always contain data in multiples of 4 bytes
				for( i = 0; i < new_packet->data_length; i += 4 )
				{
					 int address;
					 
					 if( new_packet->address_type == ADDRESS_TYPE_CONFIG )
					 {
						  address = new_packet->address - CONFIG_REG_START_ADDRESS + address_offset;
						  gConfig.r[address] = ((uint32_t)new_packet->packet_data[i+3]) | ((uint32_t)new_packet->packet_data[i+2] << 8) | ((uint32_t)new_packet->packet_data[i+1] << 16) | ((uint32_t)new_packet->packet_data[i] << 24);
					 }
					 else
					 {
						  address = new_packet->address - DATA_REG_START_ADDRESS + address_offset;
						  gData.r[address] = ((uint32_t)new_packet->packet_data[i+3]) | ((uint32_t)new_packet->packet_data[i+2] << 8) | ((uint32_t)new_packet->packet_data[i+1] << 16) | ((uint32_t)new_packet->packet_data[i] << 24);
					 }				 
					 
					 address_offset++;
				}
		  }
		  // If the packet is performing a read operation, construct a new packet containing the data to be transmitted
		  else
		  {
				SendGlobalData( new_packet->address, new_packet->address_type, (new_packet->PT & PACKET_IS_BATCH), (new_packet->PT >> 2) & PACKET_BATCH_LENGTH_MASK );
		  }
	 }
	 
	 // Now dispatch the packet for additional processing
	 DispatchPacket( new_packet );
	 
}

/*******************************************************************************
* Function Name  : SendGlobalData
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Constructs a packet containing the specified data and sends it over the USART
						 
*******************************************************************************/
void SendGlobalData(uint8_t address, uint8_t address_type, uint8_t packet_is_batch, uint8_t batch_size)
{
	 USARTPacket response_packet;
	 int i;
	 
	 response_packet.PT = PACKET_HAS_DATA;
	 response_packet.address = address;
	 
	 // If this is a batch read, then define the packet accordingly
	 if( packet_is_batch )
	 {
		  response_packet.PT |= PACKET_IS_BATCH;
		  response_packet.PT |= (batch_size << 2);
		  response_packet.data_length = 4*batch_size;
		  
		  for( i = 0; i < batch_size; i++ )
		  {
				// Check to determine whether this is a configuration register access or a data register access
				if( address_type == ADDRESS_TYPE_CONFIG )
				{
					 response_packet.packet_data[4*i] = (uint8_t)((gConfig.r[address - CONFIG_REG_START_ADDRESS + i] >> 24) & 0x0FF);
					 response_packet.packet_data[4*i+1] = (uint8_t)((gConfig.r[address - CONFIG_REG_START_ADDRESS + i] >> 16) & 0x0FF);
					 response_packet.packet_data[4*i+2] = (uint8_t)((gConfig.r[address - CONFIG_REG_START_ADDRESS + i] >> 8) & 0x0FF);
					 response_packet.packet_data[4*i+3] = (uint8_t)(gConfig.r[address - CONFIG_REG_START_ADDRESS + i] & 0x0FF);
				}
				else
				{
					 response_packet.packet_data[4*i] = (uint8_t)((gData.r[address - DATA_REG_START_ADDRESS + i] >> 24) & 0x0FF);
					 response_packet.packet_data[4*i+1] = (uint8_t)((gData.r[address - DATA_REG_START_ADDRESS + i] >> 16) & 0x0FF);
					 response_packet.packet_data[4*i+2] = (uint8_t)((gData.r[address - DATA_REG_START_ADDRESS + i] >> 8) & 0x0FF);
					 response_packet.packet_data[4*i+3] = (uint8_t)(gData.r[address - DATA_REG_START_ADDRESS + i] & 0x0FF);
				}
										
		  }
	 }
	 // If this is not a batch read, just transmit the data found in the specified address
	 else
	 {
		  response_packet.data_length = 4;
		  
		  // Check to determine whether this is a configuration register access or a data register access
		  if( address_type == ADDRESS_TYPE_CONFIG )
		  {
				response_packet.packet_data[0] = (uint8_t)((gConfig.r[address - CONFIG_REG_START_ADDRESS] >> 24) & 0x0FF);
				response_packet.packet_data[1] = (uint8_t)((gConfig.r[address - CONFIG_REG_START_ADDRESS] >> 16) & 0x0FF);
				response_packet.packet_data[2] = (uint8_t)((gConfig.r[address - CONFIG_REG_START_ADDRESS] >> 8) & 0x0FF);
				response_packet.packet_data[3] = (uint8_t)(gConfig.r[address - CONFIG_REG_START_ADDRESS] & 0x0FF);
		  }
		  else
		  {
				response_packet.packet_data[0] = (uint8_t)((gData.r[address - DATA_REG_START_ADDRESS] >> 24) & 0x0FF);
				response_packet.packet_data[1] = (uint8_t)((gData.r[address - DATA_REG_START_ADDRESS] >> 16) & 0x0FF);
				response_packet.packet_data[2] = (uint8_t)((gData.r[address - DATA_REG_START_ADDRESS] >> 8) & 0x0FF);
				response_packet.packet_data[3] = (uint8_t)(gData.r[address - DATA_REG_START_ADDRESS] & 0x0FF);
		  }
	 }
	 
	 // The response packet should now be filled with data.  Compute the Checksum and transmit the packet
	 response_packet.checksum = ComputeChecksum( &response_packet );

	 SendTXPacketSafe( &response_packet );
	 
}

/*******************************************************************************
* Function Name  : DispatchPacket
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Handles processing specific to individual packets.   This function is called
by ProcessPacket after all requisite read and write operation are performed.
						 
*******************************************************************************/
void DispatchPacket( USARTPacket* new_packet )
{
	 USARTPacket response_packet;
	 uint32_t returnval;
	 fConvert itof;
	 
	 // If this packet wrote new data to the device, copy that data to global structures for convenience
	 if( new_packet->PT & PACKET_HAS_DATA )
	 {
		  // If this packet wrote to the MISC_CONFIG register, it is possible that the estimation mode changed (from quaternions to Euler Angles or vice versa)
		  // Check to see if there was a change.  If so, reset the EKF.
		  if( ((gConfig.r[UM6_MISC_CONFIG] & UM6_QUAT_ESTIMATE_ENABLED) == 0) && (gEKF_mode == EKF_MODE_QUAT) )
		  {
				gEKF_mode = EKF_MODE_EULER;
				EKF_Init( &gStateData );				
		  }
		  else if( ((gConfig.r[UM6_MISC_CONFIG] & UM6_QUAT_ESTIMATE_ENABLED) != 0) && (gEKF_mode == EKF_MODE_EULER) )
		  {
				gEKF_mode = EKF_MODE_QUAT;
				EKF_Init( &gStateData );
		  }
		  
		  CopyConfigArrayToStates();		  
	 }
	 
	 switch(new_packet->address)
	 {
		  case UM6_GET_FW_VERSION:
				response_packet.PT = PACKET_HAS_DATA;
				response_packet.address = UM6_GET_FW_VERSION;
				response_packet.data_length = 4;
				response_packet.packet_data[0] = (UM6_FIRMWARE_REVISION >> 24) & 0x0FF;
				response_packet.packet_data[1] = (UM6_FIRMWARE_REVISION >> 16) & 0x0FF;
				response_packet.packet_data[2] = (UM6_FIRMWARE_REVISION >> 8) & 0x0FF;
				response_packet.packet_data[3] = UM6_FIRMWARE_REVISION & 0x0FF;
		  
				response_packet.checksum = ComputeChecksum( &response_packet );
		  
				SendTXPacket( &response_packet );
		  break;
		  
		  case UM6_COMMUNICATION:
				if( new_packet->PT & PACKET_HAS_DATA )
				{
					 // Update broadcast frequency
					 if( gConfig.r[UM6_COMMUNICATION] & UM6_BROADCAST_ENABLED )
					 {
						  EnableBroadcastMode( (uint8_t)(gConfig.r[UM6_COMMUNICATION] & UM6_SERIAL_RATE_MASK) );
					 }
					 else
					 {
						  DisableBroadcastMode();
					 }
				
					 // Update serial baud rate
					 UpdateSerialBaud();
					 
					 SendCommandSuccessPacket( UM6_COMMUNICATION );
				}
		  break;
		  
		  case UM6_ZERO_GYROS:
				SendCommandSuccessPacket( new_packet->address );					
				StartGyroCalibration();
		  break;
		  
		  case UM6_RESET_EKF:
				EKF_Init( &gStateData );
		  
				SendCommandSuccessPacket( new_packet->address );				
		  break;
		  
		  case UM6_GET_DATA:
				SendDataPacket();
		  break;
		  
		  case UM6_SET_ACCEL_REF:				
				gStateData.accel_ref_x = gStateData.accel_x;  
				gStateData.accel_ref_y = gStateData.accel_y;
				gStateData.accel_ref_z = gStateData.accel_z;
		  
				itof.float_val = gStateData.accel_ref_x;
				gConfig.r[UM6_ACCEL_REF_X] = itof.uint32_val;
		  
				itof.float_val = gStateData.accel_ref_y;
				gConfig.r[UM6_ACCEL_REF_Y] = itof.uint32_val;
		  
				itof.float_val = gStateData.accel_ref_z;
				gConfig.r[UM6_ACCEL_REF_Z] = itof.uint32_val;
		  
				SendCommandSuccessPacket( new_packet->address );
		  break;
		  
		  case UM6_SET_MAG_REF:
				gStateData.mag_ref_x = gStateData.mag_x;  
				gStateData.mag_ref_y = gStateData.mag_y;
				gStateData.mag_ref_z = gStateData.mag_z;
		  
				itof.float_val = gStateData.mag_ref_x;
				gConfig.r[UM6_MAG_REF_X] = itof.uint32_val;
		  
				itof.float_val = gStateData.mag_ref_y;
				gConfig.r[UM6_MAG_REF_Y] = itof.uint32_val;
		  
				itof.float_val = gStateData.mag_ref_z;
				gConfig.r[UM6_MAG_REF_Z] = itof.uint32_val;
		  
				SendCommandSuccessPacket( new_packet->address );
		  break;
		  
		  case UM6_FLASH_COMMIT:
				
				SendCommandSuccessPacket( new_packet->address );
		  
				returnval = WriteConfigurationToFlash( UM6_USE_CONFIG_ADDRESS );
		  
				if( returnval != FLASH_COMPLETE )
				{
					 SendCommandFailedPacket( new_packet->address );
				}
		  break;
				
		  case UM6_SAVE_FACTORY:
				
				SendCommandSuccessPacket( new_packet->address );
		  
				returnval = WriteConfigurationToFlash( UM6_USE_FACTORY_ADDRESS );
		  
				if( returnval != FLASH_COMPLETE )
				{
					 SendCommandFailedPacket( new_packet->address );
				}
				
		  case UM6_RESET_TO_FACTORY:
				
				SendCommandSuccessPacket( new_packet->address );
		  
				ResetToFactory();
				
		  default:
				// If this was a write command, send a "success" packet to signal that the write was succesfull
				// A read packet does not need this because the returned data packet will fill the role.  Similarly,
				// a command packet does not need it because commands are handled above
				if( new_packet->PT & PACKET_HAS_DATA )
				{
					 SendCommandSuccessPacket( new_packet->address );
				}
		  break;
		  
	 }
	 
}

/*******************************************************************************
* Function Name  : SendCommandSuccessPacket
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Sends a packet signifying that the command with address 'address' was completed
properly.
						 
*******************************************************************************/
void SendCommandSuccessPacket( uint8_t address )
{
	 USARTPacket packet;
	 
	 packet.PT = PACKET_NO_DATA;
	 packet.address = address;
	 packet.data_length = 0;
	 packet.checksum = ComputeChecksum( &packet );
	 
	 SendTXPacketSafe( &packet );	 
}

/*******************************************************************************
* Function Name  : SendCommandFailedPacket
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Sends a packet signifying that the command with address 'address' was failed
to complete properly.
						 
*******************************************************************************/
void SendCommandFailedPacket( uint8_t address )
{
	 USARTPacket packet;
	 
	 packet.PT = PACKET_NO_DATA | PACKET_COMMAND_FAILED;
	 packet.address = address;
	 packet.data_length = 0;
	 packet.checksum = ComputeChecksum( &packet );
	 
	 SendTXPacketSafe( &packet );
}

/*******************************************************************************
* Function Name  : SendStatusPacket
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Sends a packet containing the contents of the status register
						 
*******************************************************************************/
void SendStatusPacket()
{
	 USARTPacket packet;
	 
	 packet.PT = PACKET_HAS_DATA;
	 packet.address = UM6_STATUS;
	 packet.data_length = 4;
	 packet.packet_data[0] = (gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] >> 24) & 0x0FF;
	 packet.packet_data[1] = (gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] >> 16) & 0x0FF;
	 packet.packet_data[2] = (gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] >> 8) & 0x0FF;
	 packet.packet_data[3] = gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] & 0x0FF;
	 
	 packet.checksum = ComputeChecksum( &packet );
	 
	 SendTXPacketSafe( &packet );
	 
	 gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] = 0;
}

/*******************************************************************************
* Function Name  : SendDataPacket
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Sends data packets corresponding to the active channels in 
						 
*******************************************************************************/
void SendDataPacket( )
{
	 USARTPacket packet;
	 
	 // Raw accelerometer data
	 if( gConfig.r[UM6_COMMUNICATION] & UM6_ACCELS_RAW_ENABLED )
	 {
		  SendGlobalData( UM6_ACCEL_RAW_XY, ADDRESS_TYPE_DATA, PACKET_IS_BATCH, 2 );
	 }
	 
	 // Raw rate gyro data
	 if( gConfig.r[UM6_COMMUNICATION] & UM6_GYROS_RAW_ENABLED )
	 {
		  SendGlobalData( UM6_GYRO_RAW_XY, ADDRESS_TYPE_DATA, PACKET_IS_BATCH, 2 );
	 }
	 
	 // Raw magnetometer data
	 if( gConfig.r[UM6_COMMUNICATION] & UM6_MAG_RAW_ENABLED )
	 {
		  SendGlobalData( UM6_MAG_RAW_XY, ADDRESS_TYPE_DATA, PACKET_IS_BATCH, 2 );
	 }
	 
	 // Processed gyro data
	 if( gConfig.r[UM6_COMMUNICATION] & UM6_GYROS_PROC_ENABLED )
	 {
		  SendGlobalData( UM6_GYRO_PROC_XY, ADDRESS_TYPE_DATA, PACKET_IS_BATCH, 2 );
	 }
	 
	 // Processed accelerometer data
	 if( gConfig.r[UM6_COMMUNICATION] & UM6_ACCELS_PROC_ENABLED )
	 {
		  SendGlobalData( UM6_ACCEL_PROC_XY, ADDRESS_TYPE_DATA, PACKET_IS_BATCH, 2 );
	 }
	 
	 // Processed magnetometer data
	 if( gConfig.r[UM6_COMMUNICATION] & UM6_MAG_PROC_ENABLED )
	 {
		  SendGlobalData( UM6_MAG_PROC_XY, ADDRESS_TYPE_DATA, PACKET_IS_BATCH, 2 );
	 }
	 
	 // Euler angle outputs
	 if( gConfig.r[UM6_COMMUNICATION] & UM6_EULER_ENABLED )
	 {
		  SendGlobalData( UM6_EULER_PHI_THETA, ADDRESS_TYPE_DATA, PACKET_IS_BATCH, 2 );
	 }
	 
	 // Quaternion outputs
	 if( gConfig.r[UM6_COMMUNICATION] & UM6_QUAT_ENABLED )
	 {
		  SendGlobalData( UM6_QUAT_AB, ADDRESS_TYPE_DATA, PACKET_IS_BATCH, 2 );
	 }
	 
	 // Covariance output
	 if( gConfig.r[UM6_COMMUNICATION] & UM6_COV_ENABLED )
	 {
		  SendGlobalData( UM6_ERROR_COV_00, ADDRESS_TYPE_DATA, PACKET_IS_BATCH, 16 );
	 } 
}
